import numpy as np


# Problem class
class BlackboxPb(object):
    '''
        A (falsely) blackbox optimization problem based on nonconvex regression.
        
        Attributes:
            n: Dimension of the parameter space
            A: m x n matrix 
            b: m-dimensional vector
            
        Method:
            fun: Function corresponding to the underlying optimization problem
    '''
    
    def __init__(self,n,rng):
        A,b,_ = datagen(n,2*n,rng)
        self.n = n
        self.A = A
        self.b = b
        
    def fun(self,x):
        return tukey(self.A,self.b,x)

# Data generation function
def datagen(n,m,rng):
    '''
    Generate a dataset amenable to nonconvex linear regression.
    
    Inputs:
        n: Dimension of the parameter space
        m: Number of data points
        rng: Random number generator
        
    Outputs:
        A: m x n matrix
        b: Real vector with m components
        z: Ground truth (n-dimensional vector)
    '''

    A = rng.multivariate_normal(np.zeros(n),np.identity(n),size=m)
    
    z = rng.multivariate_normal(np.zeros(n),4*np.identity(n))
    
    nu1 = rng.multivariate_normal(np.zeros(m),np.identity(m))

    nu2 = rng.binomial(1,0.3,m)
    
    b = A.dot(z) + 3*nu1 + nu2
    
    return A,b,z


### True Tukey loss function
def rho(t):
    """
        Computation of the true Tukey loss function.
        
        Inputs:
            t: real number
            
        Output: 
            Loss function at t
    """
    if (abs(t)<=1):
        return (t**6/6 - t**4/2 + t**2/2)
    else:
        return 1/6

### 
def tukey(A,b,x):
    """
        Computation of the true Tukey loss function for a 
        linear regression problem.
        
        Inputs:
            A: m by n matrix
            b: m-dimensional data vector
            x: argument of the loss function (d-dimensional)
            
        Output: 
            Empirical loss value at x
    """
    m = A.shape[0]
    res = 0
    for i in range(m):
        res+=rho(A[i,:].dot(x)-b[i])
    return (res/m)


# Problem class with hidden constraints
class BlackboxPbHiddenCons(object):
    '''
        A (falsely) blackbox optimization problem based on nonconvex regression.
        
        Attributes:
            n: Dimension of the parameter space
            A: m x n matrix 
            b: m-dimensional vector
            
        Method:
            fun: Function corresponding to the underlying optimization problem
    '''
    
    def __init__(self,n,rng):
        A,b,z = datagen(n,2*n,rng)
        self.n = n
        self.A = A
        self.b = b
        self.z = z
        
    def fun(self,x):
        #if np.zeros(self.n).dot(x) >=0:
        if self.z.dot(x)>=0:
            return tukey(self.A,self.b,x)
        else:
            return np.inf