import sys
from collections import deque

# ---------- Graph Construction ---------- #

def sparseDAG(n):
    """Return a sparse DAG (assumes n is multiple of 3)."""
    dag = [[] for _ in range(n)]
    for i in range(n // 3 - 1):
        for j in range(3):
            dag[i + j * (n // 3)].extend([i + 1, i + 1 + n // 3, i + 1 + 2 * (n // 3)])
    return dag


def denseDAG(n):
    """Return a dense DAG of type 2."""
    dag = [[] for _ in range(n)]
    for i in range(n):
        if i % 7 in (1, 3):  # A
            dag[i].append(2)
            for j in range(3, n):
                if j % 7 in (2, 5):
                    dag[i].append(j)
        if i % 7 in (2, 5):  # B
            dag[i].append(0)
            for j in range(1, n):
                if j % 7 in (0, 4):
                    dag[i].append(j)
        if i % 7 in (0, 4):  # C
            dag[i].append(6)
            for j in range(7, n):
                if j % 7 == 6:
                    dag[i].append(j)
    return dag


# ---------- Topological Sorting ---------- #

def topo_sort_naive(dag):
    """Naive topological sort (O(n^2 * m))."""
    n = len(dag)
    active = [True] * n
    rank = [-1] * n
    curr = 0
    while curr < n:
        for i in range(n):
            if not active[i]:
                continue
            # check if i is a source
            is_source = True
            for j in range(n):
                if not active[j]:
                    continue
                if i in dag[j]:
                    is_source = False
                    break
            if not is_source:
                continue
            rank[i] = curr
            curr += 1
            active[i] = False
            break
    return rank


def topo_sort_degrees(dag):
    """Kahn's algorithm (O(n+m))."""
    n = len(dag)
    indeg = [0] * n
    for u in range(n):
        for v in dag[u]:
            indeg[v] += 1
    q = deque([i for i in range(n) if indeg[i] == 0])
    rank = [-1] * n
    cur = 0
    while q:
        u = q.popleft()
        rank[u] = cur
        cur += 1
        for v in dag[u]:
            indeg[v] -= 1
            if indeg[v] == 0:
                q.append(v)
    return rank


def dfs_visit(dag, i, color, cur, rank):
    color[i] = 1
    for v in dag[i]:
        if color[v] == 0:
            dfs_visit(dag, v, color, cur, rank)
    color[i] = 2
    rank[i] = cur[0]
    cur[0] -= 1


def dfs_sort(dag):
    n = len(dag)
    color = [0] * n
    rank = [-1] * n
    cur = [n - 1]
    for i in range(n):
        if color[i] == 0:
            dfs_visit(dag, i, color, cur, rank)
    return rank


# ---------- Helpers ---------- #

def sort_by_rank(rank):
    n = len(rank)
    sorted_vertices = [0] * n
    for i in range(n):
        sorted_vertices[rank[i]] = i
    return sorted_vertices


def check_ranks(dag, rank):
    n = len(dag)
    if sorted(rank) != list(range(n)):
        print("Not a valid permutation!")
        return
    for u in range(n):
        for v in dag[u]:
            if rank[u] > rank[v]:
                print(f"arc {u}->{v} violated!")
                return
    print("Looks OK")


def print_adjacency_list(dag):
    for i, neighbors in enumerate(dag):
        print(f"{i}: {neighbors}")


# ---------- Main Example ---------- #
if __name__ == "__main__":
    n = 4000000
    dag = sparseDAG(n)  # or sparseDAG(n)
   
    sys.setrecursionlimit(10000000)

    #print("Adjacency List:")
    #print_adjacency_list(dag)

    #print("\nNaive Topological Sort:")
    #rank = topo_sort_naive(dag)
    #print(sort_by_rank(rank))
    #check_ranks(dag, rank)

    #print("\nKahn's Algorithm:")
    #rank = topo_sort_degrees(dag)
    #print(sort_by_rank(rank))
    #check_ranks(dag, rank)

    print("\nDFS-based Topological Sort:")
    rank = dfs_sort(dag)
    #print(sort_by_rank(rank))
    #check_ranks(dag, rank)

