from collections import deque

# ---------- Graph Construction ---------- #

def gaber_galil(n):
    """Return adjacency dict for Gaber-Galil graph on an n x n grid, keyed by (i,j)."""
    g = {(i, j): [] for i in range(n) for j in range(n)}
    for i in range(n):
        for j in range(n):
            # (x,y) -> (x+1,y)
            g[(i, j)].append(((i + 1) % n, j))
            # (x,y) -> (x-1,y)
            g[(i, j)].append(((i - 1 + n) % n, j))
            # (x,y) -> (x,y+1)
            g[(i, j)].append((i, (j + 1) % n))
            # (x,y) -> (x,y-1)
            g[(i, j)].append((i, (j - 1 + n) % n))
            # (x,y) -> (x+y,y)
            g[(i, j)].append(((i + j) % n, j))
            # (x,y) -> (x,x+y)
            g[(i, j)].append((i, (i + j) % n))
            # (x,y) -> (x-y,y)
            g[(i, j)].append(((i - j + n) % n, j))
            # (x,y) -> (x,y-x)
            g[(i, j)].append((i, (j - i + n) % n))
    return g


# ---------- BFS ---------- #

def bfs(g, n):
    """Run BFS from (0,0), return distance dict keyed by (i,j)."""
    dist = {v: -1 for v in g}
    start = (0, 0)
    dist[start] = 0
    q = deque([start])
    while q:
        cur = q.popleft()
        for v in g[cur]:
            if dist[v] == -1:
                dist[v] = dist[cur] + 1
                q.append(v)
    return dist


# ---------- Main Example ---------- #
if __name__ == "__main__":
    import sys

    n = 100
    if len(sys.argv) == 2:
        n = int(sys.argv[1])
    g = gaber_galil(n)

    dist = bfs(g, n)
    max_dist = max(dist.values())
    print(f"For n={n}, N={n*n}, max distance is {max_dist}")
